// Copyright 2017 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.devtools.build.lib.rules.platform;

import static com.google.common.collect.ImmutableList.toImmutableList;

import com.google.common.collect.ImmutableList;
import com.google.devtools.build.lib.actions.ActionConflictException;
import com.google.devtools.build.lib.analysis.ConfiguredTarget;
import com.google.devtools.build.lib.analysis.FileProvider;
import com.google.devtools.build.lib.analysis.FilesToRunProvider;
import com.google.devtools.build.lib.analysis.RuleConfiguredTargetBuilder;
import com.google.devtools.build.lib.analysis.RuleConfiguredTargetFactory;
import com.google.devtools.build.lib.analysis.RuleContext;
import com.google.devtools.build.lib.analysis.RunfilesProvider;
import com.google.devtools.build.lib.analysis.config.ConfigMatchingProvider;
import com.google.devtools.build.lib.analysis.platform.ConstraintValueInfo;
import com.google.devtools.build.lib.analysis.platform.DeclaredToolchainInfo;
import com.google.devtools.build.lib.analysis.platform.PlatformProviderUtils;
import com.google.devtools.build.lib.analysis.platform.ToolchainTypeInfo;
import com.google.devtools.build.lib.cmdline.Label;
import com.google.devtools.build.lib.packages.BuildType;
import com.google.devtools.build.lib.packages.Type;
import javax.annotation.Nullable;

/** Defines a toolchain that can be used by rules. */
public class Toolchain implements RuleConfiguredTargetFactory {

  @Override
  @Nullable
  public ConfiguredTarget create(RuleContext ruleContext)
      throws InterruptedException, RuleErrorException, ActionConflictException {

    ToolchainTypeInfo toolchainType =
        PlatformProviderUtils.toolchainType(
            ruleContext.getPrerequisite(ToolchainRule.TOOLCHAIN_TYPE_ATTR));
    ImmutableList<ConstraintValueInfo> execConstraints =
        PlatformProviderUtils.constraintValues(
            ruleContext.getPrerequisites(ToolchainRule.EXEC_COMPATIBLE_WITH_ATTR));
    ImmutableList<ConstraintValueInfo> targetConstraints =
        PlatformProviderUtils.constraintValues(
            ruleContext.getPrerequisites(ToolchainRule.TARGET_COMPATIBLE_WITH_ATTR));
    ImmutableList<ConfigMatchingProvider> targetSettings =
        ruleContext.getPrerequisites(ToolchainRule.TARGET_SETTING_ATTR).stream()
            .map(target -> target.getProvider(ConfigMatchingProvider.class))
            .collect(toImmutableList());
    Label toolchainLabel =
        ruleContext.attributes().get(ToolchainRule.TOOLCHAIN_ATTR, BuildType.NODEP_LABEL);
    boolean targetToExecConstraints =
        ruleContext
            .attributes()
            .get(ToolchainRule.USE_TARGET_PLATFORM_CONSTRAINTS_ATTR, Type.BOOLEAN);
    if (targetToExecConstraints && !(execConstraints.isEmpty() && targetConstraints.isEmpty())) {
      ruleContext.attributeError(
          ToolchainRule.USE_TARGET_PLATFORM_CONSTRAINTS_ATTR,
          "Cannot set use_target_platform_constraints to True and also set exec_compatible_with or "
              + "target_compatible_with");
      return null;
    }

    DeclaredToolchainInfo registeredToolchain;
    try {
      var registeredToolchainBuilder =
          DeclaredToolchainInfo.builder()
              .toolchainType(toolchainType)
              .addTargetSettings(targetSettings)
              .toolchainLabel(toolchainLabel);
      if (targetToExecConstraints) {
        registeredToolchain = registeredToolchainBuilder.buildWithTargetToExecConstraints();
      } else {
        registeredToolchain =
            registeredToolchainBuilder
                .addExecConstraints(execConstraints)
                .addTargetConstraints(targetConstraints)
                .build();
      }
    } catch (DeclaredToolchainInfo.DuplicateConstraintException e) {
      if (e.execConstraintsException() != null) {
        ruleContext.attributeError(
            ToolchainRule.EXEC_COMPATIBLE_WITH_ATTR, e.execConstraintsException().getMessage());
      }
      if (e.targetConstraintsException() != null) {
        ruleContext.attributeError(
            ToolchainRule.TARGET_COMPATIBLE_WITH_ATTR, e.targetConstraintsException().getMessage());
      }
      // One of the above must have been non-null, so we just return early.
      return null;
    }

    return new RuleConfiguredTargetBuilder(ruleContext)
        .addProvider(RunfilesProvider.class, RunfilesProvider.EMPTY)
        .addProvider(FileProvider.class, FileProvider.EMPTY)
        .addProvider(FilesToRunProvider.class, FilesToRunProvider.EMPTY)
        .addProvider(registeredToolchain)
        .build();
  }
}
